<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Carbon;

use PDOException;

class MCOMUNController extends Controller{
    public function create(Request $request){
        $validator = Validator::make($request->all(), [
            'rfc_unidad' => 'required|string|max:13',
            'nombre' => 'required|string|max:100',
            'rfc_empresa' => 'required|string',
            'id_usuario_registra' => 'required|string|max:50'
        ]);

        if ($validator->fails()) {
            return $this->makeResponse(
                true,
                "ERR_MCOMUN_USU_CRE000: Se encontraron uno o más errores.",
                $this->makeErrors($validator->errors()->messages()),
                400
            );
        }

        $unidadNegocio = $request->all();
        $pdo = DB::connection()->getPdo();

        $qry = 'INSERT INTO S001V01TUNNE (UNNE_RFCU, UNNE_RFCE, UNNE_NOUN, UNNE_EMRE, UNNE_FEHR, UNNE_FEAR) 
                VALUES (:rfcu, :rfce, :noun, :emre, :fehr, CURRENT_TIMESTAMP)';
        $gst = $pdo->prepare($qry);

        $encriptacion = new EncController();
        $rfce = $encriptacion->desencriptar($unidadNegocio['rfc_empresa']);
        $hoy = Carbon::now()->timezone('America/Mexico_City')->toDateTimeString();

        $gst->bindParam(":rfcu", $unidadNegocio['rfc_unidad']);
        $gst->bindParam(":rfce", $rfce);
        $gst->bindParam(":noun", $unidadNegocio['nombre']);
        $gst->bindParam(":emre", $unidadNegocio['id_usuario_registra']);
        $gst->bindParam(":fehr", $hoy);

        try{
            if (!$gst->execute()) {
                return $this->makeResponse(true, "ERR_MCOMUN_SER_CRE001: Hubo un error al insertar el registro a la base.", [], 500);
            }
        }catch(PDOException $e){
            return $this->makeResponse(true, "ERR_MCOMUN_SER_CRE002: El rfc de la unidad ya está registrado en la base.", [], 500);
        }

        return $this->makeResponse(false, "EXITO: Unidad de negocio insertada");
    }

    public function update(Request $request){
        $validator = Validator::make($request->all(), [
            'rfc_unidad' => 'required|string|max:13',
            'nombre' => 'required|string|max:100',
            'id_usuario_modifica' => 'required|string|max:50'
        ]);

        if ($validator->fails()) {
            return $this->makeResponse(
                true,
                "ERR_MCOMUN_USU_UPD000: Se encontraron uno o más errores.",
                $this->makeErrors($validator->errors()->messages()),
                400
            );
        }

        $unidadNegocio = $request->all();
        $pdo = DB::connection()->getPdo();

        $qryRFCE = "SELECT USUA_RFCE FROM S001V01TUSUA WHERE USUA_IDUS = :idus";
        $gstRFCE = $pdo->prepare($qryRFCE);
        $gstRFCE->bindParam(":idus", $unidadNegocio['id_usuario_modifica']);

        $gstRFCE->execute();
        $rfce = $gstRFCE->fetchObject()->USUA_RFCE;

        $qry = 'UPDATE S001V01TUNNE SET UNNE_NOUN = :noun, UNNE_EMMO = :emmo, UNNE_FEHM = :fehm, UNNE_FEAR = CURRENT_TIMESTAMP 
                WHERE UNNE_RFCU = :rfcu AND UNNE_RFCE = :rfce';
        $gst = $pdo->prepare($qry);

        $hoy = Carbon::now()->timezone('America/Mexico_City')->toDateTimeString();

        $gst->bindParam(":noun", $unidadNegocio['nombre']);
        $gst->bindParam(":emmo", $unidadNegocio['id_usuario_modifica']);
        $gst->bindParam(":fehm", $hoy);
        $gst->bindParam(":rfcu", $unidadNegocio['rfc_unidad']);
        $gst->bindParam(":rfce", $rfce);

        if (!$gst->execute()) {
            return $this->makeResponse(true, "ERR_MCOMUN_SER_UPD001: Hubo un error al modificar el registro a la base", [], 500);
        }

        return $this->makeResponse(false, "EXITO: Unidad de negocio modificada");
    }

    public function delete(Request $request){
        $validator = Validator::make($request->all(), [
            'rfc_unidad' => 'required|string|max:13',
            'id_usuario_elimina' => 'required|string|max:50'
        ]);

        if ($validator->fails()) {
            return $this->makeResponse(
                true,
                "ERR_MCOMUN_USU_DEL000: Se encontraron uno o más errores",
                $this->makeErrors($validator->errors()->messages()),
                400
            );
        }

        $unidadNegocio = $request->all();
        $pdo = DB::connection()->getPdo();

        $qryRFCE = "SELECT USUA_RFCE FROM S001V01TUSUA WHERE USUA_IDUS = :idus";
        $gstRFCE = $pdo->prepare($qryRFCE);
        $gstRFCE->bindParam(":idus", $unidadNegocio['id_usuario_elimina']);

        $gstRFCE->execute();
        $rfce = $gstRFCE->fetchObject()->USUA_RFCE;

        $qryUni = "SELECT * FROM S001V01TUSUA WHERE USUA_UNNE = :rfun AND USUA_RFCE = :rfce";
        $gstUni = $pdo->prepare($qryUni);

        $gstUni->bindParam(":rfun", $unidadNegocio['rfc_unidad']);
        $gstUni->bindParam(":rfce", $rfce);

        if(!$gstUni->execute()){
            return $this->makeResponse(true, "ERR_MCOMUN_SER_DEL001: Hubo un error al consultar los usuarios con la unidad requerida", [], 500);
        }

        if(count($gstUni->fetchAll($pdo::FETCH_ASSOC)) > 0){
            return $this->makeResponse(true, "ERR_MCOMUN_USU_DEL002: La unidad de negocio solicitada no puede eliminarse porque hay usuarios registados con ella", [], 401);
        }

        $qry = 'UPDATE S001V01TUNNE SET UNNE_ESTA = "Eliminado", UNNE_EMMO = :emmo, UNNE_FEHM = :fehm, UNNE_FEAR = CURRENT_TIMESTAMP 
                WHERE UNNE_RFCU = :rfcu AND UNNE_RFCE = :rfce';
        $gst = $pdo->prepare($qry);

        $hoy = Carbon::now()->timezone('America/Mexico_City')->toDateTimeString();

        $gst->bindParam(":emmo", $unidadNegocio['id_usuario_elimina']);
        $gst->bindParam(":fehm", $hoy);
        $gst->bindParam(":rfcu", $unidadNegocio['rfc_unidad']);
        $gst->bindParam(":rfce", $rfce);

        if (!$gst->execute()) {
            return $this->makeResponse(true, "ERR_MCOMUN_SER_DEL003: Hubo un error al eliminar el registro de la base", [], 500);
        }

        return $this->makeResponse(false, "EXITO: Unidad de negocio eliminada");
    }

    public function getUnidadesNegocio($rfce, $nonce){
        $pdo = DB::connection()->getPdo();
        $encController = new EncController();
        $rfce = $encController->desencriptar(base64_encode($rfce . "|" . $nonce));
        $qry = 'SELECT UNNE_RFCU AS RFCUNIDAD, UNNE_NOUN AS NOMBREUNIDAD FROM S001V01TUNNE WHERE UNNE_ESTA  != "Eliminado" AND UNNE_RFCE = :rfce';
        $gst = $pdo->prepare($qry);
        $gst->bindParam(":rfce", $rfce);
        
        if (!$gst->execute()) {
            return $this->makeResponse(true, "ERR_MCOMUN_SER_GUN000: Hubo un error al recuperar las unidades", [], 500);
        }

        return $this->makeResponse(false, "EXITO", $gst->fetchAll($pdo::FETCH_ASSOC));
    }

    public function getUnidadNegocio($id){   
        $pdo = DB::connection()->getPdo();
        $qry = 'SELECT * FROM mcomun WHERE id = :id';
        $gst = $pdo->prepare($qry);

        $gst->bindParam(":id", $id);
        
        if (!$gst->execute()) {
            //return response()->json(['error' => 'error_get_unidad_negocio'], 500);
            return $this->makeResponse(true, "Hubo un error al buscar el registro en la base", [], 500);
        }

        $unidadNegocio = $gst->fetchObject();

        if(!$unidadNegocio){
            return $this->makeResponse(true, "La unidad de negocio solicitada no existe", [], 404);
        }

        if ($unidadNegocio->estatus == 'E') {
            //return response()->json(['error' => 'Unidad de negocio eliminada'], 404);
            return $this->makeResponse(true, "Unidad de negocio eliminada", [], 404);
        }

        //return response()->json(['unidad_negocio' => $unidadNegocio], 200);
        return $this->makeResponse(false, "Consulta exitosa", $unidadNegocio);
    }

    private function makeResponse($error, $msg, $response = [], $code = 200){
        $respuesta = json_encode([
            "error" => $error,
            "msg" => $msg,
            "response" => $response
        ]);

        return response($respuesta, $code)->header('Content-Type', 'application/json');
    }

    private function makeErrors($erroresObj){
        $erroresArr = array();

        foreach ($erroresObj as $key => $value) {
            foreach ($value as $key0 => $value0) {
                if(array_key_exists($key, $erroresArr)){
                    $val = $erroresArr[$key] . "|" . $value0;
                    $erroresArr[$key] = $val;
                }else{
                    $erroresArr[$key] = $value0;
                }
            }
        }

        return $erroresArr;
    }
}
